const fs = require('fs');
const path = require('path');
const moment = require('moment-timezone');

class MediaHandler {
  constructor() {
    this.baseDir = path.join(__dirname, '..', 'arquivos');
    this.photosDir = path.join(this.baseDir, 'fotos');
    this.videosDir = path.join(this.baseDir, 'videos');
    this.adminDir = path.join(this.baseDir, 'admin');
    
    this.ensureDirectories();
  }

  /**
   * Create directories if they don't exist
   */
  ensureDirectories() {
    [this.baseDir, this.photosDir, this.videosDir, this.adminDir].forEach(dir => {
      if (!fs.existsSync(dir)) {
        fs.mkdirSync(dir, { recursive: true });
        console.log(`✅ Directory created: ${dir}`);
      }
    });
  }

  /**
   * Save media and return path
   * @param {Object} message - WPP message
   * @param {string} phone - Phone number
   * @param {string} name - Client/agent name
   * @param {boolean} isAdmin - If admin is sending
   * @returns {Promise<string>} - Saved file path
   */
  async saveMedia(client, message, phone, name, isAdmin = false) {
    try {
      // Download media
      const buffer = await client.decryptFile(message);
      
      // Determine type and extension
      const mediaType = message.type;
      const mimetype = message.mimetype || '';
      const extension = this.getExtension(mimetype, mediaType);
      
      // Generate filename
      const timestamp = moment().format('YYYYMMDD-HHmmss');
      const cleanPhone = phone.replace(/[^0-9]/g, '');
      const cleanName = name.replace(/[^a-zA-Z0-9]/g, '');
      
      let filename;
      let targetDir;
      
      if (isAdmin) {
        // Admin: admin/number-date.ext
        filename = `${cleanPhone}-${timestamp}.${extension}`;
        targetDir = this.adminDir;
      } else {
        // Client: fotos/number-name-date.ext or videos/number-name-date.ext
        filename = `${cleanPhone}-${cleanName}-${timestamp}.${extension}`;
        targetDir = this.isVideo(mediaType, mimetype) ? this.videosDir : this.photosDir;
      }
      
      const filepath = path.join(targetDir, filename);
      
      // Save file
      fs.writeFileSync(filepath, buffer);
      
      console.log(`✅ Media saved: ${filepath}`);
      return filepath;
      
    } catch (error) {
      console.error('❌ Error saving media:', error.message);
      return null;
    }
  }

  /**
   * Determine if it's a video
   */
  isVideo(type, mimetype) {
    return type === 'video' || mimetype.startsWith('video/');
  }

  /**
   * Get file extension
   */
  getExtension(mimetype, type) {
    // MIME types mapping
    const mimeMap = {
      'image/jpeg': 'jpg',
      'image/jpg': 'jpg',
      'image/png': 'png',
      'image/gif': 'gif',
      'image/webp': 'webp',
      'video/mp4': 'mp4',
      'video/mpeg': 'mpeg',
      'video/quicktime': 'mov',
      'video/x-msvideo': 'avi',
      'audio/mpeg': 'mp3',
      'audio/ogg': 'ogg',
      'application/pdf': 'pdf',
      'application/msword': 'doc',
      'application/vnd.openxmlformats-officedocument.wordprocessingml.document': 'docx'
    };

    if (mimeMap[mimetype]) {
      return mimeMap[mimetype];
    }

    // Fallback based on type
    const typeMap = {
      'image': 'jpg',
      'video': 'mp4',
      'audio': 'mp3',
      'ptt': 'ogg',
      'document': 'pdf'
    };

    return typeMap[type] || 'bin';
  }

  /**
   * List files of a client
   */
  listClientMedia(phone) {
    const cleanPhone = phone.replace(/[^0-9]/g, '');
    const allFiles = [];

    [this.photosDir, this.videosDir].forEach(dir => {
      if (fs.existsSync(dir)) {
        const files = fs.readdirSync(dir)
          .filter(f => f.startsWith(cleanPhone))
          .map(f => path.join(dir, f));
        allFiles.push(...files);
      }
    });

    return allFiles;
  }
}

module.exports = new MediaHandler();
